/*******************************************************************************
 * Copyright (c) 2017 Rene Schneider, GEBIT Solutions GmbH and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package de.gebit.integrity.runner.time;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.temporal.TemporalUnit;
import java.util.List;

import org.eclipse.xtext.util.Pair;

/**
 * An object encapsulating the necessary internal parameters for fake test time generation.
 *
 * @author Rene Schneider - initial API and implementation
 *
 */
public class TimeSyncState {

	/**
	 * The offset from realtime, in msecs since 1.1.1970.
	 */
	private long realtimeOffset;

	/**
	 * The time at which the fake time was decoupled from realtime, in msecs since 1.1.1970.
	 */
	private long realtimeDecouplingTime;

	/**
	 * A time progression factor (1.0 = realtime, 0.0 = frozen).
	 */
	private double progressionFactor;

	/**
	 * This instance represents live time.
	 */
	public static final TimeSyncState LIVE = new TimeSyncState();

	/**
	 * Constructor.
	 * 
	 * @param aRealtimeOffset
	 *            The offset from realtime, in msecs since 1.1.1970.
	 * @param aRealtimeDecouplingTime
	 *            The time at which the fake time was decoupled from realtime, in msecs since 1.1.1970.
	 * @param aProgressionFactor
	 *            A time progression factor (1.0 = realtime, 0.0 = frozen).
	 */
	public TimeSyncState(long aRealtimeOffset, long aRealtimeDecouplingTime, double aProgressionFactor) {
		this.realtimeOffset = aRealtimeOffset;
		this.realtimeDecouplingTime = aRealtimeDecouplingTime;
		this.progressionFactor = aProgressionFactor;
	}

	/**
	 * Constructor using a data string previously generated by {@link #getValuesAsString()}.
	 *
	 * @param anInternalsString
	 *            the string to use as source
	 */
	public TimeSyncState(String anInternalsString) {
		if (anInternalsString == null || "null".equals(anInternalsString)) {
			this.realtimeOffset = 0;
			this.realtimeDecouplingTime = 0;
			this.progressionFactor = 0.0;
		} else {
			String[] tempParts = anInternalsString.split("\\|");
			if (tempParts.length != 3) {
				throw new IllegalArgumentException(
						"Unexpected date/time singleton internals string formatting: " + anInternalsString);
			}
			try {
				this.realtimeOffset = Long.parseLong(tempParts[0]);
				this.realtimeDecouplingTime = Long.parseLong(tempParts[1]);
				this.progressionFactor = new BigDecimal(tempParts[2]).doubleValue();
			} catch (NumberFormatException exc) {
				throw new IllegalArgumentException(
						"Unexpected date/time singleton internals string formatting: " + anInternalsString, exc);
			}
		}
	}

	/**
	 * Constructs an instance that represents local live time.
	 */
	public TimeSyncState() {
		this.realtimeOffset = 0;
		this.realtimeDecouplingTime = System.currentTimeMillis();
		this.progressionFactor = 1.0;
	}

	public double getProgressionFactor() {
		return progressionFactor;
	}

	public long getRealtimeDecouplingTime() {
		return realtimeDecouplingTime;
	}

	public long getRealtimeOffset() {
		return realtimeOffset;
	}

	/**
	 * Returns the values of this {@link TimeSyncState} as a String. This string is intended to be fed into another
	 * {@link TimeSyncState} in order to synchronize it with the source. The string consists of pipe-divided values
	 * "realtime offset", "realtime decoupling time" and "multiplier" (first two are long values, third is a
	 * double-precision floating point number in normal notation with decimal point).
	 *
	 * @return the internal values as string without spaces
	 */
	public String getValuesAsString() {
		StringBuilder tempBuilder = new StringBuilder();
		tempBuilder.append(realtimeOffset);
		tempBuilder.append("|");
		tempBuilder.append(realtimeDecouplingTime);
		tempBuilder.append("|");
		tempBuilder.append(new BigDecimal(progressionFactor).toPlainString());

		return tempBuilder.toString();
	}

	/**
	 * Returns a new instance of this {@link TimeSyncState} to which the given duration was added. This does NOT modify
	 * the state that it is called upon!
	 * 
	 * @param aDiffTime
	 *            the time to add (negative = subtract)
	 * @return a new {@link TimeSyncState} instance
	 */
	public TimeSyncState plus(List<Pair<Long, TemporalUnit>> aDiffTime) {
		// We need to know the current date/time in order to add the duration, because some specifications of a duration
		// (like in months or years) actually result in different lengths of time, depending on the current date (for
		// example the length of a month differs depending on the month).
		long tempTimeSinceDecoupling = System.currentTimeMillis() - realtimeDecouplingTime;
		long tempStartingTimeMillis = realtimeDecouplingTime + realtimeOffset
				+ (long) (tempTimeSinceDecoupling * progressionFactor);
		LocalDateTime tempCurrentTime = Instant.ofEpochMilli(tempStartingTimeMillis).atZone(ZoneId.systemDefault())
				.toLocalDateTime();

		for (Pair<Long, TemporalUnit> tempPair : aDiffTime) {
			tempCurrentTime = tempCurrentTime.plus(tempPair.getFirst(), tempPair.getSecond());
		}

		long tempResultingTimeMillis = tempCurrentTime.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
		long tempTimeDifference = tempResultingTimeMillis - tempStartingTimeMillis;

		return new TimeSyncState(realtimeOffset + tempTimeDifference, realtimeDecouplingTime, progressionFactor);
	}

	/**
	 * Calculates the current date/time according to the states' parameters.
	 * 
	 * @return
	 */
	public ZonedDateTime calculateCurrentZonedDateTime() {
		long tempTimeSinceDecoupling = System.currentTimeMillis() - realtimeDecouplingTime;
		long tempStartingTimeMillis = realtimeDecouplingTime + realtimeOffset
				+ (long) (tempTimeSinceDecoupling * progressionFactor);
		return Instant.ofEpochMilli(tempStartingTimeMillis).atZone(ZoneId.systemDefault());
	}

	/**
	 * Calculates the current date/time according to the states' parameters.
	 * 
	 * @return
	 */
	public LocalDateTime calculateCurrentDateTime() {
		return calculateCurrentZonedDateTime().toLocalDateTime();
	}

	@Override
	public String toString() {
		return "realtimeOffset=" + realtimeOffset + ", realtimeDecouplingTime=" + realtimeDecouplingTime
				+ ", progressionFactor=" + progressionFactor;
	}

}
